<?php

namespace Modules\Blog\Repositories;

use App\Enums\OrderBy;
use App\Traits\ApiReturnFormatTrait;
use App\Traits\FileUploadTrait;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Modules\Blog\Entities\Career;
use Modules\Blog\Interfaces\CareerInterface;
use Illuminate\Support\Facades\Storage;

class CareerRepository implements CareerInterface
{
    use ApiReturnFormatTrait, FileUploadTrait;

    private $model;

    public function __construct(Career $careerModel)
    {
        $this->model = $careerModel;
    }

    public function getAll($request)
    {

        try {

            $data = $this->model->with('imageDetail');

            $data = $this->filter($request, $data);

            $data = $data->orderBy('id', OrderBy::DESC)->paginate($request->show ?? 10);

            return $data;

        } catch (\Throwable $th) {
            // dd($th);
            return false;
        }

    }

    public function filter($request, $data)
    {

        if (!empty($request->search)) {
            $data = $data->where('title', 'like', '%' . $request->search . '%');
        }

        return $data;
    }

    public function all()
    {
        try {
            return $this->model->get();

        } catch (\Throwable $th) {
            return false;
        }

    }

    public function tableHeader()
    {

        return [
            ___('common.ID'),
            ___('common.Title'),
            ___('common.Image'),
            ___('ui_element.action'),
        ];
    }

    public function model()
    {
        try {
            return $this->model;

        } catch (\Throwable $th) {
            return false;
        }
    }

    public function store($request)
    {

        DB::beginTransaction(); // start database transaction
        try {

            if (env('APP_DEMO')) {
                return $this->responseWithError(___('alert.you_can_not_change_in_demo_mode'));
            }

            $career = new $this->model; // create new object of model for store data in database table
            $career->name = $request->name ?? '';
            $career->title = $request->title ?? '';
            $career->overview = $request->overview ?? '';
            $career->eligibility = $request->eligibility ?? '';
            $career->job_roles = $request->job_roles ?? '';
            $career->salary = $request->salary ?? '';
            $career->career_opportunity = $request->career_opportunity ?? '';
            $career->top_colleges = $request->top_colleges ?? '' ;
            
            if ($request->hasFile('image')) {
                $upload = $this->uploadFile($request->image, 'blog/image/', [[35, 35], [300, 220], [750, 400]], '', 'image'); // upload file and resize image 35x35
                // dd($upload);
                if ($upload['status']) {
                    $career->image = $upload['upload_id'];
                } else {
                    return $this->responseWithError($upload['message'], [], 400);
                }
            }

            // dd($career);
            $career->save(); // save data in database table
            DB::commit(); // commit database transaction
            return $this->responseWithSuccess(___('alert.Career created successfully.')); // return success response
        } catch (\Throwable $th) {
            DB::rollBack(); // rollback database transaction
            return $this->responseWithError($th->getMessage(), [], 400); // return error response
        }
    }

    public function show($id)
    {
        try {

            return $this->model->with('imageDetail')->find($id);

        } catch (\Throwable $th) {
            return false;
        }

    }

    public function update($request, $id)
    {
        DB::beginTransaction();
        try {

            if (env('APP_DEMO')) {
                return $this->responseWithError(___('alert.you_can_not_change_in_demo_mode'));
            }


            $career = $this->model->find($id);
            if (!$career) {
                return $this->responseWithError(___('alert.Career not found.'), [], 400);
            }
            $career->name = $request->name ?? '';
            $career->title = $request->title ?? '';
            $career->overview = $request->overview ?? '';
            $career->eligibility = $request->eligibility ?? '';
            $career->job_roles = $request->job_roles ?? '';
            $career->salary = $request->salary ?? '';
            $career->career_opportunity = $request->career_opportunity ?? '';
            $career->top_colleges = $request->top_colleges ?? '';
            
            if ($request->hasFile('image')) {
                if (Storage::exists('public/' . $career->image)) {
                    Storage::delete('public/' . $career->image);
                }
                $upload = $this->uploadFile($request->image, 'blog/image/', [[35, 35], [300, 220], [750, 400]], '', 'image'); // upload file and resize image 35x35
                // dd($upload);
                if ($upload['status']) {
                    $career->image = $upload['upload_id'];
                } else {
                    return $this->responseWithError($upload['message'], [], 400);
                }
            }
            $career->save();
            DB::commit();
            return $this->responseWithSuccess(___('alert.Career updated successfully.'));
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->responseWithError($th->getMessage(), [], 400);
        }
    }

    public function destroy($id)
    {
        try {

            if (env('APP_DEMO')) {
                return $this->responseWithError(___('alert.you_can_not_change_in_demo_mode'));
            }

            $target = $this->model->find($id);

            $upload = $this->deleteFile($target->image, 'delete'); // delete file from storage
            if (!$upload['status']) {
                return $this->responseWithError($upload['message'], [], 400); // return error response
            }

            $target->delete();

            return $this->responseWithSuccess(___('alert.Career deleted successfully.')); // return success response
        } catch (\Throwable $th) {
            return $this->responseWithError($th->getMessage(), [], 400); // return error response
        }
    }

    public function getCareers()
    {
        try {

            return $this->model->query()->with('imageDetail')->latest()->paginate(8);

        } catch (\Throwable $th) {

            return false;
        }

    }

    // Use this function at Home page career section
    public function homeCareer()
    {
        try {

            return $this->model->query()->with('imageDetail')->latest()->take(4)->get();

        } catch (\Throwable $th) {
            dd($th);
            return false;

        }

    }

}
