<?php

namespace Modules\Organization\Repositories;

use App\Enums\Role;
use App\Events\AdminEmailVerificationEvent;
use App\Events\UserEmailVerifyEvent;
use App\Models\Country;
use App\Models\User;
use Illuminate\Support\Facades\Storage;
use App\Traits\ApiReturnFormatTrait;
use App\Traits\FileUploadTrait;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;
use Modules\Organization\Entities\Organization;
use Modules\Organization\Interfaces\OrganizationInterface;

class OrganizationRepository implements OrganizationInterface
{
    use ApiReturnFormatTrait, FileUploadTrait;

    private $organizationModel;
    private $countryModel;
    protected $userModel;

    public function __construct(Organization $organizationModel, User $user, Country $countryModel)
    {
        $this->organizationModel = $organizationModel;
        $this->countryModel = $countryModel;
        $this->userModel = $user;
    }

    public function model()
    {
        return $this->organizationModel;
    }
    
    
    public function filter($request)
    {
        $query = $this->model()->active();
    
        \Log::info('Filtering organizations', $request->all());
    
        if ($request->has('state') && $request->state) {
            $states = explode(',', $request->state);
            $query = $query->whereIn('state', $states);
        }
        
         if ($request->has('city') && $request->city) {
            $cities = explode(',', $request->city);
            $query = $query->whereIn('city', $cities);
        }
    
        // Add more filters as needed
    
        $result = $query->get();
        \Log::info('SQL Query: ' . $query->toSql());
    
        return $query;
    }
    
    public function activeAndFilter($request)
    {
        return $this->model()->active()->filter($request);
    }

// public function filter($filter = null)
//     {
//         $model = $this->model();
//         if (@$filter) {
//             $model = $this->model()->where($filter);
//         }
//         return $model;
//     }

    public function create($request)
    {
        DB::beginTransaction(); // start database transaction
        try {
            if (env('APP_DEMO')) {
                return $this->responseWithError(___('alert.you_can_not_change_in_demo_mode'));
            }
            $user = $this->userModel->where('email', $request->email)->first();
            if ($user) {
                return $this->responseWithError(___('alert.Email already exists'), [], 400);
            }
            $user = $this->userModel;
            $user->name = $request->name;
            $user->username = Str::slug($request->name);
            $user->email = $request->email;
            $user->token = Str::random(30);
            $user->password = Hash::make($request->password);
            $user->phone = $request->phone;
            $user->role_id = 6;
            $user->status_id = 4;
            $user->save();

            $request->session()->put('password', $request->password); // store user id in session for store data in database table

            $organization = $this->organizationModel; // create new object of model for store data in database table
            $organization->user_id = $user->id;
            $organization->save();

            $alert = ___('alert.Please check email to verify this account.');
            try {
                event(new AdminEmailVerificationEvent($user));
            } catch (\Throwable $th) {
                $alert = ___('alert.Organization create but please configure SMTP to send email correctly');
            }
            $request->session()->forget('password'); // remove user id from session

            DB::commit(); // commit database transaction
            return $this->responseWithSuccess($alert, [], 200);
        } catch (\Throwable $th) {
            DB::rollBack(); // rollback database transaction
            return $this->responseWithError($th->getMessage(), [], 400);
        }
    }

    public function suspend($id)
    {
        DB::beginTransaction(); // start database transaction
        try {
            if (env('APP_DEMO')) {
                return $this->responseWithError(___('alert.you_can_not_change_in_demo_mode'));
            }
            $organization = $this->model()->where('id', $id)->first();
            $user = $organization->user;
            $user->status_id = 5;
            $user->save();
            DB::commit(); // commit database transaction
            return $this->responseWithSuccess(___('alert.Organization suspended successfully'), [], 200);
        } catch (\Throwable $th) {
            DB::rollBack(); // rollback database transaction
            return $this->responseWithError($th->getMessage(), [], 400);
        }
    }

    public function reActivate($id)
    {
        DB::beginTransaction(); // start database transaction
        try {
            if (env('APP_DEMO')) {
                return $this->responseWithError(___('alert.you_can_not_change_in_demo_mode'));
            }
            $organization = $this->model()->where('id', $id)->first();
            $user = $organization->user;
            $user->status_id = 4;
            $user->save();
            DB::commit(); // commit database transaction
            return $this->responseWithSuccess(___('alert.Organization re-activate successfully'), [], 200);
        } catch (\Throwable $th) {
            DB::rollBack(); // rollback database transaction
            return $this->responseWithError($th->getMessage(), [], 400);
        }
    }

    public function approve($id)
    {
        DB::beginTransaction(); // start database transaction
        try {
            if (env('APP_DEMO')) {
                return $this->responseWithError(___('alert.you_can_not_change_in_demo_mode'));
            }
            $organization = $this->model()->where('id', $id)->first();
            $user = $organization->user;
            $user->status_id = 4;
            $user->save();
            DB::commit(); // commit database transaction
            return $this->responseWithSuccess(___('alert.Organization approved successfully'), [], 200);
        } catch (\Throwable $th) {
            DB::rollBack(); // rollback database transaction
            return $this->responseWithError($th->getMessage(), [], 400);
        }
    }

    public function updateProfile($request, $id)
    {
        DB::beginTransaction(); // start database transaction
        try {
            if (env('APP_DEMO')) {
                return $this->responseWithError(___('alert.you_can_not_change_in_demo_mode'));
            }
            // dd($request->all());
            $organizationModel = $this->organizationModel->where('user_id', $id)->first(); // create new object of model for store data in database table
            $organizationModel->address = $request->address;
            // $organizationModel->country_id = $request->country_id;
            $organizationModel->city = $request->city;
            $organizationModel->state = $request->state;

            $organizationModel->established_year = $request->established_year;
            $organizationModel->website = $request->website;
            $organizationModel->about_me = $request->about_me;

            $organizationModel->designation = $request->designation;
            $organizationModel->save(); // save data in database table
            $user = $organizationModel->user;
            if ($request->hasFile('profile_image')) {
                // if (Storage::exists('public/' . $user->image->original)) {
                //     Storage::delete('public/' . $user->image->original);
                // }
                // dd($request->all());
                $upload = $this->uploadFile($request->profile_image, 'organization/profile', [], $user->image_id, 'image'); // upload file and resize image 35x35
                // dd($upload);
                if ($upload['status']) {
                    $user->image_id = $upload['upload_id'];
                } else {
                    return $this->responseWithError($upload['message'], [], 400);
                }
            }
            // dd($request->all());
            if ($request->hasFile('cover_image')) {
                if (Storage::exists('public/' . $organizationModel->cover_image)) {
                    Storage::delete('public/' . $organizationModel->cover_image);
                }
                $upload = $this->uploadFile($request->cover_image, 'organization/cover', [], $organizationModel->cover_image, 'image'); // upload file and resize image 35x35
                // dd($upload);
                if ($upload['status']) {
                    $organizationModel->cover_image = $upload['upload_id'];
                    // dd($organizationModel);
                } else {
                    return $this->responseWithError($upload['message'], [], 400);
                }
            }
            
            if ($request->hasFile('thumbnail_image')) {
                if (Storage::exists('public/' . $organizationModel->thumbnail_image)) {
                    Storage::delete('public/' . $organizationModel->thumbnail_image);
                }
                $uploadThumnail = $this->uploadFile($request->thumbnail_image, 'organization/thumbnail', [], $organizationModel->thumbnail_image, 'image');
                //dd($uploadThumnail);
                if ($uploadThumnail['status']) {
                    $organizationModel->thumbnail_image = $uploadThumnail['upload_id'];
                } else {
                    return $this->responseWithError($uploadThumnail['message'], [], 400);
                }
            }

            if ($request->hasFile('brochure')) {
                if (Storage::exists('public/' . $organizationModel->brochure_id)) {
                    Storage::delete('public/' . $organizationModel->brochure_id);
                }
                $uploadBrochure = $this->uploadFile($request->brochure, 'organization/brochure', [], $organizationModel->brochure_id, 'image');
                //dd($uploadBrochure);
                if ($uploadBrochure['status']) {
                    $organizationModel->brochure_id = $uploadBrochure['upload_id'];
                } else {
                    return $this->responseWithError($uploadBrochure['message'], [], 400);
                }
            }

            $organizationModel->organization_type = $request->organization_type;
            $organizationModel->slug = Str($request->name) . '-' . Str::random(8);
            // dd($organizationModel);
            $organizationModel->save();


            $user->name = $request->name;
            $user->phone = $request->phone;
            $user->save();
            DB::commit(); // commit database transaction
            return $this->responseWithSuccess(___('alert.Profile updated successfully.')); // return success response
        } catch (\Throwable $th) {
            DB::rollBack(); // rollback database transaction
            return $this->responseWithError($th->getMessage(), [], 400); // return error response
        }
    }

    public function updatePassword($request, $user)
    {
        DB::beginTransaction(); // start database transaction
        try {
            if (env('APP_DEMO')) {
                return $this->responseWithError(___('alert.you_can_not_change_in_demo_mode'));
            }
            if (!Hash::check($request->old_password, $user->password)) {
                return $this->responseWithError(___('alert.Old password does not match.'), [], 400);
            }
            $user->password = Hash::make($request->password);
            $user->save();

            DB::commit(); // commit database transaction
            return $this->responseWithSuccess(___('alert.Password updated successfully.')); // return success response
        } catch (\Throwable $th) {
            DB::rollBack(); // rollback database transaction
            return $this->responseWithError($th->getMessage(), [], 400); // return error response
        }
    }

    public function delete($id)
    {
        DB::beginTransaction(); // start database transaction
        try {
            if (env('APP_DEMO')) {
                return $this->responseWithError(___('alert.you_can_not_change_in_demo_mode'));
            }
            $organization = $this->model()->find($id);
            $organization->user->delete();
            DB::commit(); // commit database transaction
            return $this->responseWithSuccess(___('alert.Organization deleted successfully.')); // return success response

        } catch (\Throwable $th) {
            DB::rollBack(); // rollback database transaction
            return $this->responseWithError($th->getMessage(), [], 400); // return error response
        }
    }

    // start skills
    public function storeSkill($request, $id)
    {

        DB::beginTransaction(); // start database transaction
        try {
            if (env('APP_DEMO')) {
                return $this->responseWithError(___('alert.you_can_not_change_in_demo_mode'));
            }
            $organizationModel = $this->organizationModel->where('user_id', $id)->first(); // create new object of model for store data in database table
            if (!@$organizationModel) {
                $organizationModel = new $this->organizationModel;
                $organizationModel->user_id = $id;
                $organizationModel->save();
            }
            // dd($request->all());
            $organizationModel->skills = json_decode($request->skills);
            $organizationModel->save(); // save data in database table
            DB::commit(); // commit database transaction
            return $this->responseWithSuccess(___('alert.Organization skills added successfully.')); // return success response
        } catch (\Throwable $th) {
            DB::rollBack(); // rollback database transaction
            return $this->responseWithError($th->getMessage(), [], 400); // return error response
        }
    }
    //for store collage info




    public function deleteCollegeInfo($key, $id,$field)
    {
        DB::beginTransaction(); // start database transaction
        try {

            if (env('APP_DEMO')) {
                return $this->responseWithError(___('alert.you_can_not_change_in_demo_mode'));
            }

            $organizationModel = $this->organizationModel->where('user_id', $id)->first(); // create new object of model for store data in database table
            if (!@$organizationModel) {
                return $this->responseWithError(___('alert.Student not found.'), [], 400);
            }

            $collageinfo = $organizationModel->$field ?? [];
            if (isset($collageinfo[$key])) {
                unset($collageinfo[$key]);
            }
            $organizationModel->$field = $collageinfo;
            $organizationModel->save(); // save data in database table
            DB::commit(); // commit database transaction
            return $this->responseWithSuccess(___('alert.'.$field.' deleted successfully.')); // return success response
        } catch (\Throwable $th) {
            DB::rollBack(); // rollback database transaction
            return $this->responseWithError(___('alert.something_went_wrong_please_try_again'), [], 400); // return error response
        }
    }


    public function update($request, $organization, $slug)
    {
        try {
            if (env('APP_DEMO')) {
                return $this->responseWithError(___('alert.you_can_not_change_in_demo_mode'));
            }
            if ($slug == 'general') {
                return $this->updateProfile($request, $organization->user_id);
            } elseif ($slug == 'security') {
                return $this->updatePassword($request, $organization->user);
            } else {
                return $this->responseWithError(___('alert.Invalid request.'), [], 400); // return error response
            }
        } catch (\Throwable $th) {
            return $this->responseWithError($th->getMessage(), [], 400); // return error response
        }
    }

    public function storeCollageInfo($request, $id)
    {
        DB::beginTransaction(); // start database transaction
        try {
            if (env('APP_DEMO')) {
                return $this->responseWithError(___('alert.you_can_not_change_in_demo_mode'));
            }
            $organizationModel = $this->organizationModel->where('user_id', $id)->first(); // create new object of model for store data in database table
            if (!@$organizationModel) {
                $organizationModel = new $this->organizationModel;
                $organizationModel->user_id = $id;
                $organizationModel->save();
            }
            // dd($request->all());

            $organizationArr = [];
            $collageinfo = $organizationModel->college_info ?? [];
            // dd($collageinfo);
            if ($request->title) {
                $organizationArr = [
                    'title' => $request->title,
                    'description' => $request->description,
                ];


                // dd($organizationArr);
                array_push($collageinfo, $organizationArr);
            }
            // dd($collageinfo);
            $organizationModel->college_info = $collageinfo;
            $organizationModel->save();

            // save data in database table
            DB::commit(); // commit database transaction
            return $this->responseWithSuccess(___('alert.College info added successfully.')); // return success response
        } catch (\Throwable $th) {
            DB::rollBack(); // rollback database transaction
            return $this->responseWithError($th->getMessage(), [], 400); // return error response
        }
    }

    public function storeCollegeLink($request, $id)
    {
        // dd($request->all());
        DB::beginTransaction(); // start database transaction
        try {
            if (env('APP_DEMO')) {
                return $this->responseWithError(___('alert.you_can_not_change_in_demo_mode'));
            }
            $organizationModel = $this->organizationModel->where('user_id', $id)->first();
            if (!@$organizationModel) {
                $organizationModel = new $this->organizationModel;
                $organizationModel->user_id = $id;
                $organizationModel->save();
            }

            $collegeLinkArr = [];
            $collegeLinks = $organizationModel->college_links ?? [];

            if ($request->title && $request->group) {
                $collegeLinkArr = [
                    'group' => $request->group,
                    'title' => $request->title,
                    
                ];

                array_push($collegeLinks, $collegeLinkArr);
            }

            $organizationModel->college_links = $collegeLinks;
            $organizationModel->save();

            DB::commit(); // commit database transaction
            return $this->responseWithSuccess(___('alert.College link added successfully.')); // return success response
        } catch (\Throwable $th) {
            DB::rollBack(); // rollback database transaction
            return $this->responseWithError($th->getMessage(), [], 400); // return error response
        }
    }

    

    public function storeCourse($request, $id)
    {
        DB::beginTransaction(); // start database transaction
        try {
            if (env('APP_DEMO')) {
                return $this->responseWithError(___('alert.you_can_not_change_in_demo_mode'));
            }
            $organizationModel = $this->organizationModel->where('user_id', $id)->first();
            if (!@$organizationModel) {
                $organizationModel = new $this->organizationModel;
                $organizationModel->user_id = $id;
                $organizationModel->save();
            }

            $courseArr = [];
            $course = $organizationModel->course ?? [];

            if ($request->title) {
                $courseArr = [
                    'title' => $request->title,
                    'description' => $request->description,
                ];

                array_push($course, $courseArr);
            }

            $organizationModel->course = $course;
            $organizationModel->save();

            DB::commit(); // commit database transaction
            return $this->responseWithSuccess(___('alert.Course  added successfully.')); // return success response
        } catch (\Throwable $th) {
            DB::rollBack(); // rollback database transaction
            return $this->responseWithError($th->getMessage(), [], 400); // return error response
        }
    }

    public function storeAdmission($request, $id)
    {
        DB::beginTransaction(); // start database transaction
        try {
            if (env('APP_DEMO')) {
                return $this->responseWithError(___('alert.you_can_not_change_in_demo_mode'));
            }
            $organizationModel = $this->organizationModel->where('user_id', $id)->first();
            if (!@$organizationModel) {
                $organizationModel = new $this->organizationModel;
                $organizationModel->user_id = $id;
                $organizationModel->save();
            }

            $admissionArr = [];
            $admission = $organizationModel->admission ?? [];

            if ($request->title) {
                $admissionArr = [
                    'title' => $request->title,
                    'description' => $request->description,
                ];

                array_push($admission, $admissionArr);
            }

            $organizationModel->admission = $admission;
            $organizationModel->save();

            DB::commit(); // commit database transaction
            return $this->responseWithSuccess(___('alert.Admission added successfully.')); // return success response
        } catch (\Throwable $th) {
            DB::rollBack(); // rollback database transaction
            return $this->responseWithError($th->getMessage(), [], 400); // return error response
        }
    }

    public function storePlacement($request, $id)
    {
        DB::beginTransaction(); // start database transaction
        try {
            if (env('APP_DEMO')) {
                return $this->responseWithError(___('alert.you_can_not_change_in_demo_mode'));
            }
            $organizationModel = $this->organizationModel->where('user_id', $id)->first();
            if (!@$organizationModel) {
                $organizationModel = new $this->organizationModel;
                $organizationModel->user_id = $id;
                $organizationModel->save();
            }

            $placementArr = [];
            $placement = $organizationModel->placement ?? [];

            if ($request->title) {
                $placementArr = [
                    'title' => $request->title,
                    'description' => $request->description,
                ];

                array_push($placement, $placementArr);
            }

            $organizationModel->placement = $placement;
            $organizationModel->save();

            DB::commit(); // commit database transaction
            return $this->responseWithSuccess(___('alert.Placement added successfully.')); // return success response
        } catch (\Throwable $th) {
            DB::rollBack(); // rollback database transaction
            return $this->responseWithError($th->getMessage(), [], 400); // return error response
        }
    }

    public function storeReviews($request, $id)
    {
        DB::beginTransaction(); // start database transaction
        try {
            if (env('APP_DEMO')) {
                return $this->responseWithError(___('alert.you_can_not_change_in_demo_mode'));
            }
            $organizationModel = $this->organizationModel->where('user_id', $id)->first();
            if (!@$organizationModel) {
                $organizationModel = new $this->organizationModel;
                $organizationModel->user_id = $id;
                $organizationModel->save();
            }

            $reviewArr = [];
            $reviews = $organizationModel->reviews ?? [];

            if ($request->title) {
                $reviewArr = [
                    'title' => $request->title,
                    'description' => $request->description,
                ];

                array_push($reviews, $reviewArr);
            }

            $organizationModel->reviews = $reviews;
            $organizationModel->save();

            DB::commit(); // commit database transaction
            return $this->responseWithSuccess(___('alert.Review added successfully.')); // return success response
        } catch (\Throwable $th) {
            DB::rollBack(); // rollback database transaction
            return $this->responseWithError($th->getMessage(), [], 400); // return error response
        }
    }

    public function storeCampus($request, $id)
    {
        DB::beginTransaction(); // start database transaction
        try {
            if (env('APP_DEMO')) {
                return $this->responseWithError(___('alert.you_can_not_change_in_demo_mode'));
            }
            $organizationModel = $this->organizationModel->where('user_id', $id)->first();
            if (!@$organizationModel) {
                $organizationModel = new $this->organizationModel;
                $organizationModel->user_id = $id;
                $organizationModel->save();
            }

            $campusArr = [];
            $campus = $organizationModel->campus ?? [];

            if ($request->title) {
                $campusArr = [
                    'title' => $request->title,
                    'description' => $request->description,
                ];

                array_push($campus, $campusArr);
            }

            $organizationModel->campus = $campus;
            $organizationModel->save();

            DB::commit(); // commit database transaction
            return $this->responseWithSuccess(___('alert.Campus added successfully.')); // return success response
        } catch (\Throwable $th) {
            DB::rollBack(); // rollback database transaction
            return $this->responseWithError($th->getMessage(), [], 400); // return error response
        }
    }

    public function storeScholarship($request, $id)
    {
        DB::beginTransaction(); // start database transaction
        try {
            if (env('APP_DEMO')) {
                return $this->responseWithError(___('alert.you_can_not_change_in_demo_mode'));
            }
            $organizationModel = $this->organizationModel->where('user_id', $id)->first();
            if (!@$organizationModel) {
                $organizationModel = new $this->organizationModel;
                $organizationModel->user_id = $id;
                $organizationModel->save();
            }

            $scholarshipArr = [];
            $scholarships = $organizationModel->scholarships ?? [];

            if ($request->title) {
                $scholarshipArr = [
                    'title' => $request->title,
                    'description' => $request->description,
                ];

                array_push($scholarships, $scholarshipArr);
            }

            $organizationModel->scholarships = $scholarships;
            $organizationModel->save();

            DB::commit(); // commit database transaction
            return $this->responseWithSuccess(___('alert.Scholarship added successfully.')); // return success response
        } catch (\Throwable $th) {
            DB::rollBack(); // rollback database transaction
            return $this->responseWithError($th->getMessage(), [], 400); // return error response
        }
    }

    public function storeGallery($request, $id)
    {
        DB::beginTransaction(); // start database transaction
        try {
            if (env('APP_DEMO')) {
                return $this->responseWithError(___('alert.you_can_not_change_in_demo_mode'));
            }
            $organizationModel = $this->organizationModel->where('user_id', $id)->first();
            if (!@$organizationModel) {
                $organizationModel = new $this->organizationModel;
                $organizationModel->user_id = $id;
                $organizationModel->save();
            }

            $galleryArr = [];
            $gallery = $organizationModel->gallery ?? [];

            if ($request->gallery_images) {

                //upload multiple images

                $gallery_images = $request->file('gallery_images');
                foreach ($gallery_images as $image) {
                    $upload = $this->uploadFile($image, 'organization/gallery', [], null, 'image'); // upload file and resize image 35x35
                    if ($upload['status']) {
                        $galleryArr = [
                            'image' => $upload['upload_id'],
                        ];
                        array_push($gallery, $galleryArr);
                    } else {
                        return $this->responseWithError($upload['message'], [], 400);
                    }
                }


              
            }

            $organizationModel->gallery = $gallery;
            $organizationModel->save();

            DB::commit(); // commit database transaction
            return $this->responseWithSuccess(___('alert.Gallery item added successfully.')); // return success response
        } catch (\Throwable $th) {
            DB::rollBack(); // rollback database transaction
            return $this->responseWithError($th->getMessage(), [], 400); // return error response
        }
    }

    public function storeAdmissionLink($request, $id)
    {
        // dd($request->all());
        DB::beginTransaction(); // start database transaction
        try {
            if (env('APP_DEMO')) {
                return $this->responseWithError(___('alert.you_can_not_change_in_demo_mode'));
            }
            $organizationModel = $this->organizationModel->where('user_id', $id)->first();
            if (!@$organizationModel) {
                $organizationModel = new $this->organizationModel;
                $organizationModel->user_id = $id;
                $organizationModel->save();
            }

            $admissionLinkArr = [];
            $admissionLinks = $organizationModel->admission_links ?? [];

            if ($request->title && $request->group) {
                $admissionLinkArr = [
                    'group' => $request->group,
                    'title' => $request->title,
                    
                ];

                array_push($admissionLinks, $admissionLinkArr);
            }
            // dd($admissionLinks);
            $organizationModel->admission_links = $admissionLinks;
            $organizationModel->save();

            DB::commit(); // commit database transaction
            return $this->responseWithSuccess(___('alert.Admission link added successfully.')); // return success response
        } catch (\Throwable $th) {
            DB::rollBack(); // rollback database transaction
            return $this->responseWithError($th->getMessage(), [], 400); // return error response
        }
    }


    public function storeLink($request, $id,$field)
    {
        // dd($request);
        DB::beginTransaction(); // start database transaction
        try {
            if (env('APP_DEMO')) {
                return $this->responseWithError(___('alert.you_can_not_change_in_demo_mode'));
            }
            $organizationModel = $this->organizationModel->where('user_id', $id)->first();
            if (!@$organizationModel) {
                $organizationModel = new $this->organizationModel;
                $organizationModel->user_id = $id;
                $organizationModel->save();
            }

            $linkArr = [];
            // dd($field);
            if($organizationModel->$field){
                $links = $organizationModel->$field;
            }else{
                $links = [];
            }

            // $links = ($organizationModel->$field) ?? [];
            // dd($links);
            if ($request->group && $request->title) {
                $linkArr = [
                    'group' => $request->group,
                    'title' => $request->title,
                ];

                array_push($links, $linkArr);
            }

            // dd($links);

            $organizationModel->$field = $links;
            $organizationModel->save();

            DB::commit(); // commit database transaction
            return $this->responseWithSuccess(___('alert.Link added successfully.')); // return success response
        } catch (\Throwable $th) {
            DB::rollBack(); // rollback database transaction
            return $this->responseWithError($th->getMessage(), [], 400); // return error response
        }
    }
    

    public function storePlacementLink($request, $id)
    {
        DB::beginTransaction(); // start database transaction
        try {
            if (env('APP_DEMO')) {
                return $this->responseWithError(___('alert.you_can_not_change_in_demo_mode'));
            }
            $organizationModel = $this->organizationModel->where('user_id', $id)->first();
            if (!@$organizationModel) {
                $organizationModel = new $this->organizationModel;
                $organizationModel->user_id = $id;
                $organizationModel->save();
            }

            $placementLinkArr = [];
            $placementLinks = $organizationModel->placement_links ?? [];

            if ($request->group && $request->title) {
                $placementLinkArr = [
                    'group' => $request->group,
                    'title' => $request->title,
                ];

                array_push($placementLinks, $placementLinkArr);
            }

            $organizationModel->placement_links = $placementLinks;
            $organizationModel->save();

            DB::commit(); // commit database transaction
            return $this->responseWithSuccess(___('alert.Placement link added successfully.')); // return success response
        } catch (\Throwable $th) {
            DB::rollBack(); // rollback database transaction
            return $this->responseWithError($th->getMessage(), [], 400); // return error response
        }
    }

    public function storeCampusLink($request, $id,$fie)
    {
        // dd($request);
        DB::beginTransaction(); // start database transaction
        try {
            if (env('APP_DEMO')) {
                return $this->responseWithError(___('alert.you_can_not_change_in_demo_mode'));
            }
            $organizationModel = $this->organizationModel->where('user_id', $id)->first();
            if (!@$organizationModel) {
                $organizationModel = new $this->organizationModel;
                $organizationModel->user_id = $id;
                $organizationModel->save();
            }

            $campusLinkArr = [];
            $campusLinks = $organizationModel->campus_links ?? [];

            if ($request->title && $request->group) {
                $campusLinkArr = [
                    'group' => $request->group,
                    'title' => $request->title,
                ];

                array_push($campusLinks, $campusLinkArr);
            }
            // dd($campusLinks);
            $organizationModel->campus_links = $campusLinks;
            $organizationModel->save();

            DB::commit(); // commit database transaction
            return $this->responseWithSuccess(___('alert.Campus link added successfully.')); // return success response
        } catch (\Throwable $th) {
            DB::rollBack(); // rollback database transaction
            return $this->responseWithError($th->getMessage(), [], 400); // return error response
        }
    }
}