<?php

namespace App\Http\Controllers\Panel\Student;

use App\Http\Controllers\Controller;
use App\Models\Modules\Student\Entities\CounselorSchedule;
use App\Models\User;
use App\Traits\ApiReturnFormatTrait;
use App\Traits\CommonHelperTrait;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Modules\Course\Interfaces\BookmarkInterface;
use Modules\Instructor\Entities\Instructor;
use Modules\Order\Interfaces\EnrollInterface;
use Modules\Order\Interfaces\NoteInterface;
use Modules\Organization\Interfaces\OrganizationInterface;
use Modules\Order\Repositories\EnrollRepository;
use Modules\Organization\Repositories\OrganizationRepository;
use Modules\Student\Interfaces\StudentInterface;
use Modules\Student\Repositories\StudentRepository;
use Modules\Instructor\Interfaces\InstructorInterface;
use Modules\Student\Entities\Feedback;
use Modules\Student\Entities\Student;
use Modules\Slot\Entities\Slot;

class StudentController extends Controller
{
    use ApiReturnFormatTrait, CommonHelperTrait;

    protected $user;
    protected $student;
    protected $enrollRepository;
    protected $organizationRepository;
    protected $organization;
    protected $instructor;
    protected $noteRepository;
    protected $bookmarkRepository;
    protected $template = 'panel.student';

    public function __construct(
        User $user,
        StudentInterface $student,
        EnrollInterface $enrollRepository,
        BookmarkInterface $bookmarkRepository,
        NoteInterface $noteRepository,
        InstructorInterface $instructorInterface,
        OrganizationInterface $organizationInterface
    ) {
        $this->user = $user;
        $this->student = $student;
        $this->enrollRepository = $enrollRepository;
        $this->instructor = $instructorInterface;
        $this->noteRepository = $noteRepository;
        $this->organization = $organizationInterface;
        $this->bookmarkRepository = $bookmarkRepository;
    }

    public function statistics()
    {
        try {
            $data['title'] = ___('instructor.Student'); // title
            $data['student'] = $this->student->model()->where('user_id', Auth::id())->first();
            $data['title'] = ___('student.Statistics'); // title
            return view($this->template . '.statistics', compact('data'));
        } catch (\Throwable $th) {
            return redirect()->back()->with('danger', ___('alert.something_went_wrong_please_try_again'));
        }
    }

    public function dashboard()
    {
        try {

            $data['student'] = $this->student->model()->where('user_id', Auth::id())->first();
            $data['title'] = ___('student.Student Dashboard'); // title
            return view($this->template . '.student_dashboard', compact('data'));
        } catch (\Throwable $th) {
            return redirect()->back()->with('danger', ___('alert.something_went_wrong_please_try_again'));
        }
    }

    public function profile()
    {
        try {
            $data['title'] = ___('student.My Profile'); // title
            $data['student'] = $this->student->model()->where('user_id', Auth::id())->first();
            return view($this->template . '.student_profile', compact('data'));
        } catch (\Throwable $th) {
            return redirect()->back()->with('danger', ___('alert.something_went_wrong_please_try_again'));
        }
    }

    public function education()
    {
        try {
            $data['title'] = ___('student.My Documents'); // title
            $data['student'] = $this->student->model()->where('user_id', Auth::id())->first();
            return view($this->template . '.student_education', compact('data'));
        } catch (\Throwable $th) {
            return redirect()->back()->with('danger', ___('alert.something_went_wrong_please_try_again'));
        }
    }

    public function courses(Request $request)
    {
        try {
            $data['title'] = ___('student.Student Courses'); // title
            $data['enrolls'] = $this->enrollRepository->model()->where('user_id', Auth::id())->with('course:id,title,course_duration,course_category_id,slug,thumbnail')
                ->search($request)
                ->latest()
                ->paginate(10);
            return view($this->template . '.my_courses', compact('data'));
        } catch (\Throwable $th) {
            return redirect()->back()->with('danger', ___('alert.something_went_wrong_please_try_again'));
        }
    }

    public function institutes(Request $request)
    {
        // try {
            $data['title'] = ___('student.Student Institutes'); // title
        $data['organizations'] = $this->organization->model()->active() // data
            ->search($request)
                ->latest()
                ->paginate(10);
            return view($this->template . '.my_institutes', compact('data'));
        // } catch (\Throwable $th) {
            // return redirect()->back()->with('danger', ___('alert.something_went_wrong_please_try_again'));
        // }
    }

    public function referEarn()
    {
        try {
            $student = auth()->id();
            $student = User::where('id', $student)->first();
            $referralLink = route('student.sign_up',$student->referral_code);

            $data['title'] = ___('student.Refer & Earn'); // title
            $data['referralLink'] = $referralLink;
            $data['referralCount'] = $student->referralCount();
            $data['referralHistory'] =  $student->referrals()->with('referred')->get();
            
            return view($this->template . '.refer_earn', compact('data'));
            
        } catch (\Throwable $th) {
            return redirect()->back()->with('danger', ___('alert.something_went_wrong_please_try_again'));
        }
    }

    public function feedback()
    {
        try {
            $data['title'] = ___('student.Feedback'); // title
            return view($this->template . '.feedback', compact('data'));
        } catch (\Throwable $th) {
            return redirect()->back()->with('danger', ___('alert.something_went_wrong_please_try_again'));
        }
    }

    public function storeFeedback(Request $request)
    {
        try {
            $request->validate([
                'feedback' => 'required'
            ]);
            $student = auth()->id();
            $student = User::where('id', $student)->first();
            Feedback::create([
                'student_id' => $student->id,
                'feedback' => $request->feedback
            ]);
            return redirect()->back()->with('success', ___('alert.Feedback submitted successfully'));
        } catch (\Throwable $th) {
            return redirect()->back()->with('danger', ___('alert.something_went_wrong_please_try_again'));
        }
    }

    public function tabLoad(Request $request)
    {
        try {
            if (@$request->tab && $request->enrollId) {
                $data['enroll'] = $this->enrollRepository->model()->find(decryptFunction($request->enrollId));
                if (!$data['enroll']) {
                    return $this->responseWithError(___('alert.something_went_wrong_please_try_again'), [], 400); // return error response
                }
                if ($request->tab == 'Notes') {
                    $view = view($this->template . '.course.tab.notes', compact('data'))->render();
                    return $this->responseWithSuccess(___('alert.Data found'), $view);
                } elseif ($request->tab == 'Announcement') {
                    $announcement = $this->enrollRepository->model()->announcements($data['enroll']);
                    return $this->responseWithSuccess(___('alert.Data found'), @$announcement);
                } elseif ($request->tab == 'Assignment') {
                    $view = view($this->template . '.course.tab.assignment', compact('data'))->render();
                    return $this->responseWithSuccess(___('alert.Data found'), $view);
                } elseif ($request->tab == 'Review') {
                    $view = view($this->template . '.course.tab.reviews', compact('data'))->render();
                    return $this->responseWithSuccess(___('alert.Data found'), $view);
                } else {
                    return $this->responseWithError(___('alert.something_went_wrong_please_try_again'), [], 400); // return error response
                }
            } else {
                return $this->responseWithError(___('alert.something_went_wrong_please_try_again'), [], 400); // return error response
            }
        } catch (\Throwable $th) {
            return $this->responseWithError(___('alert.something_went_wrong_please_try_again'), [], 400); // return error response
        }
    }

    public function courseLearn($slug, $lesson_id)
    {
        try {
            $lesson_id = decryptFunction($lesson_id);
            $data['title'] = ___('student.Student Course Learn'); // title
            $data['enroll'] = $this->enrollRepository->model()->where('user_id', Auth::id())->whereHas('course', function ($q) use ($slug) {
                $q->where('slug', $slug);
            })->with('course:id,title,slug,course_duration,created_by,requirements,outcomes,description', 'lessons')->first();
            $data['lesson'] = $data['enroll']->lessons->find($lesson_id);
            if (!$data['enroll'] || !$data['lesson']) {
                return redirect()->back()->with('danger', ___('alert.Lesson not found'));
            }
            $this->enrollRepository->visited($data['enroll']);
            $data['lesson_id'] = $lesson_id;
            return view($this->template . '.course.course_details', compact('data'));
        } catch (\Throwable $th) {
            return redirect()->back()->with('danger', ___('alert.something_went_wrong_please_try_again'));
        }
    }

    public function courseEnrollProgress(Request $request)
    {
        try {
            $result = $this->enrollRepository->update($request);
            if ($result->original['result']) {
                return $this->responseWithSuccess(___('alert.Lesson_successfully_updated'));
            } else {
                return $this->responseWithError($result->original['message'], [], 400); // return error response
            }
        } catch (\Throwable $th) {
            return $this->responseWithError(___('alert.something_went_wrong_please_try_again'), [], 400); // return error response
        }
    }

    // course activities
    public function courseActivities(Request $request)
    {
        try {
            $data['enrolls'] = $this->enrollRepository->model()->where('user_id', Auth::id())->with('course:id,title,course_duration,point,course_category_id,slug')
                ->search($request)
                ->latest()
                ->paginate(10);
            $data['title'] = ___('student.Course Activities'); // title
            return view($this->template . '.course_activities', compact('data'));
        } catch (\Throwable $th) {
            return redirect()->back()->with('danger', ___('alert.something_went_wrong_please_try_again'));
        }
    }
    // course activities

    public function leaderBoard()
    {
        try {
            $data['title'] = ___('student.Leaderboard'); // title
            $data['students'] = $this->student->model()->orderBy('points', 'DESC')->paginate(10);
            return view($this->template . '.leader_board', compact('data'));
        } catch (\Throwable $th) {
            return redirect()->back()->with('danger', ___('alert.something_went_wrong_please_try_again'));
        }
    }

    public function logout()
    {
        try {
            auth()->logout();

            return redirect()->route('home')->with('success', ___('alert.Student Log out successfully!!'));
        } catch (\Throwable $th) {
            return redirect()->back()->with('danger', ___('alert.something_went_wrong_please_try_again'));
        }
    }


    public function uploadDocuments(Request $request, StudentRepository $studentRepository)
    {
        $request->validate([
            'aadhaar_card' => 'required|file|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'pan_card' => 'nullable|file|mimes:jpeg,png,jpg,gif,svg|max:2048',
        ]);

        $student = Auth::user()->student;

        if ($request->hasFile('aadhaar_card')) {
            $upload = $studentRepository->uploadFile($request->aadhaar_card, 'student/documents', [], '', 'image');
            // dd($upload);
            if ($upload['status']) {
                $student->aadhaar_card = $upload['upload_id'];
            } else {
                return redirect()->back()->with('danger', $upload['message']);
            }
        }

        if ($request->hasFile('pan_card')) {
            $upload = $studentRepository->uploadFile($request->pan_card, 'student/documents', [], '', 'image');
            if ($upload['status']) {
                $student->pan_card = $upload['upload_id'];
            } else {
                return redirect()->back()->with('danger', $upload['message']);
            }
        }

        $student->save();

        return redirect()->back()->with('success', 'Documents uploaded successfully.');
    }

    public function talkToCounselor(Request $request, StudentRepository $studentRepository)
    {
        // try {
            $data['title'] = ___('student.Talk to Our Expert Counselors'); // title
            $student = auth()->user()->id;
            // dd($student);
            $student = $this->student->model()->where('user_id', Auth::id())->first();
            $scheduledCalls = $student->scheduledCalls()->with('instructor.user')->get();
            $data['instructors'] = $this->instructor->model()->with(['user'])->whereHas('user', function ($query) {
                $query->where('status', 1);
            })->paginate(10);
            $data['scheduledCalls'] = $scheduledCalls;

            return view($this->template . '.talk_to_counselor', compact('data'));
        // } catch (\Throwable $th) {
        //     return redirect()->back()->with('danger', ___('alert.something_went_wrong_please_try_again'));
        // }
    }

    public function scheduleCall($id)
    {

        try {
            $data['instructor'] = Instructor::find($id);
            $data['url'] = route('student.setScheduleCall'); // url
            $data['title'] = ___('course.Schedule Call'); // title
            @$data['button'] = ___('common.Submit'); // button
            $slots = Slot::where('user_id', $data['instructor']->user_id)->get();
            $data['slots'] = $slots;
            $html = view('panel.student.partials.modal.counselor.schedule_call', compact('data'))->render(); // render view
            // dd($data);
            return $this->responseWithSuccess(___('alert.data_retrieve_success'), $html); // return success response
        } catch (\Throwable $th) {
            return $this->responseWithError(___('alert.something_went_wrong_please_try_again'), [], 400); // return error response
        }
    }

    public function reviewCounselor($id)
    {
        $data['instructor'] = Instructor::find($id);
        $data['url'] = route('student.storeReviewCounselor'); // url
        $data['title'] = ___('student.Review Counselor'); // title
        @$data['button'] = ___('common.Submit'); // button
        $html = view('panel.student.partials.modal.counselor.review_counselor', compact('data'))->render(); // render view
        return $this->responseWithSuccess(___('alert.data_retrieve_success'), $html); // return success response
    }

    public function setScheduleCall(Request $request, StudentRepository $studentRepository)
    {
        try {
            $result = $studentRepository->setScheduleCall($request, auth()->user()->id);
            if ($result->original['result']) {
                return redirect()->back()->with('success', $result->original['message']);
            } else {
                // dd($result->original['message']);
                return redirect()->back()->with('danger', $result->original['message']);
            }
        } catch (\Throwable $th) {
            return redirect()->back()->with('danger', ___('alert.something_went_wrong_please_try_again'));
            
        }
        
    }

    private function getCounselors()
    {
        // Implement logic to fetch counselors
        return [
            ['name' => 'John Doe', 'email' => 'john@example.com'],
            ['name' => 'Jane Smith', 'email' => 'jane@example.com'],
        ];
    }


    public function showLevels()
    {
        // Assuming the authenticated user
        $user = User::find(auth()->id());
        $data['title'] = ___('instructor.Levels'); // title
        // Determine the user's level based on points
        $level = $this->determineUserLevel($user->points);

        $income = $this->calculateIncome($user->points);

        return view('panel.student.levels', compact('user', 'level', 'income', 'data'));
    }

    private function calculateIncome($points)
    {
        return ($points / 100) * 50;
    }

    private function determineUserLevel($points)
    {
        if ($points <= 10) {
            return 'Well Wisher';
        } elseif ($points <= 500) {
            return 'Bronze Certificate';
        } elseif ($points <= 1000) {
            return 'Silver Certificate';
        } elseif ($points <= 5000) {
            return 'Gold Certificate';
        } else {
            return 'Platinum Certificate';
        }
    }

    //redeemPoints
    public function redeemPoints()
    {
        try {
            $data['title'] = ___('instructor.Redeem Points'); // title

            return view('panel.student.redeem_points', compact('data'));
        } catch (\Throwable $th) {
            return redirect()->back()->with('danger', ___('alert.something_went_wrong_please_try_again'));
        }
    }

    public function redeemPointsStore(Request $request)
    {
        // Validate the request

        $validator = Validator::make($request->all(), [
            'points_to_redeem' => 'required|integer|min:1|max:' . auth()->user()->points,
        ]);

        if ($validator->fails()) {
            return redirect()->back()->with('danger', ___($validator->errors()->first()));
        }

        // Assuming the authenticated user
        $user = User::find(auth()->id());

        // Define the points to be redeemed and the corresponding amount
        $pointsToRedeem = $request->input('points_to_redeem');
        $amountPerPoint = 0.5; // 100 points = ₹50, so 1 point = ₹0.5

        // Calculate the total amount
        $totalAmount = $pointsToRedeem * $amountPerPoint;

        // Deduct the points from the user's balance
        $user->points -= $pointsToRedeem;
        $user->balance += $totalAmount;
        $user->save();

        // Add the redeemed amount to the user's balance (or handle as needed)
        // Example: $user->balance += $totalAmount;
        // $user->save();

        return redirect()->back()->with('success', 'Points redeemed successfully. You have received ₹' . number_format($totalAmount, 2));
    }
}